import { useState, useCallback, useEffect, useRef } from "react";

// ─── Config ────────────────────────────────────────────────────────────────────
const ROWS = 20, COLS = 20; // Square grid like competitor
const CELL = 40;        // px per cell
const PAD  = 20;        // board padding
const SW   = 3.2;       // stroke width
const NAVY = "#0d1a4a";

// Edge offset: how far from cell center the line runs
// Lines run at CELL*0.25 from center = along inner edge boundary
// This gives clean parallel lines with gap between them
const EDGE = CELL * 0.25;

const DX   = { U:0,  D:0,  L:-1, R:1  };
const DY   = { U:-1, D:1,  L:0,  R:0  };
const OPP  = { U:"D",D:"U",L:"R",R:"L"};
const DIRS = ["U","D","L","R"];

function mv(r,c,d){ return [r+DY[d], c+DX[d]]; }
function inB(r,c) { return r>=0&&r<ROWS&&c>=0&&c<COLS; }
function ckey(r,c){ return `${r},${c}`; }

// ─── SVG coordinate helpers ────────────────────────────────────────────────────
// Cell (r,c) → its grid origin corner (top-left)
function gx(c){ return PAD + c*CELL; }
function gy(r){ return PAD + r*CELL; }

// ─── Edge-based path routing ───────────────────────────────────────────────────
// Each arrow is routed along cell edges, NOT cell centers.
// Strategy: for each segment direction, the line runs at a fixed offset
// from the shared cell wall. We pick one "lane" per arrow based on its id.
// 
// For an arrow with cells [tail...head] and exitDir:
// - The line runs along the INTERIOR of each cell it occupies
// - At corners, it makes a sharp 90° turn
// - The position within the cell is determined by the TRAVEL DIRECTION of each segment
//
// Lane assignment: each arrow gets lane 0 (near edge) based on direction:
//   Horizontal segment → line runs at y = gy(r) + CELL/2 ± EDGE
//   Vertical segment   → line runs at x = gx(c) + CELL/2 ± EDGE
// The ± sign is based on arrow id to separate parallel arrows.

function getSegmentDirs(cells) {
  const dirs = [];
  for (let i = 0; i < cells.length - 1; i++) {
    const [r1,c1] = cells[i], [r2,c2] = cells[i+1];
    const dr = r2-r1, dc = c2-c1;
    dirs.push(dr===-1?"U": dr===1?"D": dc===-1?"L": "R");
  }
  return dirs;
}

// Build path connecting grid points (dots), not cell centers
// Arrows follow the grid structure - paths connect grid intersections (dots)
function buildPath(cells, exitDir/*, lane*/) {
  if (!cells.length) return [];

  const pts = [];
  
  if (cells.length === 1) {
    // Single cell: path goes from one grid corner to opposite corner
    const [r, c] = cells[0];
    // Tail at corner opposite exit, head at corner in exit direction
    const tailX = gx(c) + (DX[exitDir] < 0 ? CELL : 0);
    const tailY = gy(r) + (DY[exitDir] < 0 ? CELL : 0);
    const headX = gx(c) + (DX[exitDir] > 0 ? CELL : DX[exitDir] < 0 ? 0 : CELL);
    const headY = gy(r) + (DY[exitDir] > 0 ? CELL : DY[exitDir] < 0 ? 0 : CELL);
    return [[tailX, tailY], [headX, headY]];
  }

  // Multi-cell: path connects grid points along cell edges
  // Head cell's corner in exit direction
  const [hr, hc] = cells[0];
  const headX = gx(hc) + (DX[exitDir] > 0 ? CELL : 0);
  const headY = gy(hr) + (DY[exitDir] > 0 ? CELL : 0);
  pts.push([headX, headY]);
  
  // For each cell, determine the grid point at the corner
  // Path follows the grid: goes through corners where cells meet
  for (let i = 0; i < cells.length - 1; i++) {
    const [r1, c1] = cells[i];
    const [r2, c2] = cells[i + 1];
    const dr = r2 - r1;
    const dc = c2 - c1;
    
    // Corner point between these two cells
    // If moving right: use right edge of cell1 = left edge of cell2
    // If moving down: use bottom edge of cell1 = top edge of cell2
    const cornerX = gx(c1) + (dc > 0 ? CELL : dc < 0 ? 0 : (DX[exitDir] > 0 ? CELL : 0));
    const cornerY = gy(r1) + (dr > 0 ? CELL : dr < 0 ? 0 : (DY[exitDir] > 0 ? CELL : 0));
    
    // Also add the corner of cell2
    const nextX = gx(c2) + (dc < 0 ? CELL : 0);
    const nextY = gy(r2) + (dr < 0 ? CELL : 0);
    
    if (i === 0 || cornerX !== pts[pts.length - 1][0] || cornerY !== pts[pts.length - 1][1]) {
      pts.push([cornerX, cornerY]);
    }
    if (nextX !== cornerX || nextY !== cornerY) {
      pts.push([nextX, nextY]);
    }
  }
  
  // Tail at last cell's corner opposite exit direction
  const [tr, tc] = cells[cells.length - 1];
  const tailX = gx(tc) + (DX[exitDir] < 0 ? CELL : 0);
  const tailY = gy(tr) + (DY[exitDir] < 0 ? CELL : 0);
  
  if (tailX !== pts[pts.length - 1][0] || tailY !== pts[pts.length - 1][1]) {
    pts.push([tailX, tailY]);
  }
  
  return pts;
}

// ─── Path helpers for snake animation ───────────────────────────────────────────
// Cumulative distances along polyline (cum[i] = distance from pts[0] to pts[i])
function pathCumulative(pts) {
  const cum = [0];
  for (let i = 0; i < pts.length - 1; i++) {
    const [x1, y1] = pts[i];
    const [x2, y2] = pts[i + 1];
    cum.push(cum[cum.length - 1] + Math.hypot(x2 - x1, y2 - y1));
  }
  return cum;
}

function pointAtDist(pts, cum, dist) {
  if (dist <= 0) return pts[0].slice();
  const total = cum[cum.length - 1];
  if (dist >= total) return pts[pts.length - 1].slice();
  for (let i = 0; i < cum.length - 1; i++) {
    if (dist >= cum[i] && dist <= cum[i + 1]) {
      const t = (cum[i + 1] - cum[i]) < 1e-6 ? 1 : (dist - cum[i]) / (cum[i + 1] - cum[i]);
      return [
        pts[i][0] + t * (pts[i + 1][0] - pts[i][0]),
        pts[i][1] + t * (pts[i + 1][1] - pts[i][1]),
      ];
    }
  }
  return pts[pts.length - 1].slice();
}

// Slice polyline from tailDist to headDist (snake-like visible segment)
function slicePath(pts, cum, tailDist, headDist) {
  const total = cum[cum.length - 1];
  if (tailDist >= headDist || headDist <= 0 || tailDist >= total) return [];
  const result = [pointAtDist(pts, cum, tailDist)];
  for (let i = 1; i < pts.length - 1; i++) {
    if (cum[i] > tailDist && cum[i] < headDist) result.push(pts[i].slice());
  }
  result.push(headDist >= total ? pts[pts.length - 1].slice() : pointAtDist(pts, cum, headDist));
  return result;
}

// ─── Arrow SVG component ───────────────────────────────────────────────────────
function ArrowShape({ arrow, color, onClick, anim }) {
  const { cells, exitDir, id } = arrow;
  const lane = (id % 2 === 0) ? 1 : -1;
  let rawPts = buildPath(cells, exitDir, lane);

  // Extend path for exit animation (snake needs path beyond grid)
  const origLen = pathCumulative(rawPts).pop();
  const exitExt = origLen + CELL * 2;
  const last = rawPts[rawPts.length - 1];
  rawPts = [...rawPts, [
    last[0] + DX[exitDir] * exitExt,
    last[1] + DY[exitDir] * exitExt,
  ]];

  const cum = pathCumulative(rawPts);
  const totalLen = cum[cum.length - 1];

  let pts;
  if (anim && anim.id === arrow.id) {
    const tailDist = anim.progress * totalLen;
    const headDist = origLen + anim.progress * totalLen;
    pts = slicePath(rawPts, cum, tailDist, headDist);
  } else {
    pts = rawPts.slice(0, -1);
  }

  if (pts.length < 2) return null;

  const HL = 13, HW = 8;
  const ddx = DX[exitDir], ddy = DY[exitDir];
  const nx = -ddy, ny = ddx;

  const [endX, endY] = pts[pts.length - 1];
  const baseX = endX;
  const baseY = endY;
  const tipX = baseX + ddx * HL;
  const tipY = baseY + ddy * HL;

  pts[pts.length - 1] = [baseX, baseY];
  const poly = pts.map(([x, y]) => `${x.toFixed(1)},${y.toFixed(1)}`).join(" ");

  const p1 = `${(baseX + nx * HW).toFixed(1)},${(baseY + ny * HW).toFixed(1)}`;
  const p2 = `${(baseX - nx * HW).toFixed(1)},${(baseY - ny * HW).toFixed(1)}`;

  return (
    <g onClick={onClick} style={{ cursor: "pointer" }}>
      <polyline points={poly} fill="none" stroke="transparent"
        strokeWidth={CELL * 0.85} strokeLinejoin="miter" strokeLinecap="square" />
      <polyline points={poly} fill="none" stroke={color}
        strokeWidth={SW} strokeLinejoin="miter" strokeLinecap="square" />
      <polygon points={`${tipX.toFixed(1)},${tipY.toFixed(1)} ${p1} ${p2}`} fill={color} />
    </g>
  );
}

// ─── Shape-based organic maze generator ──────────────────────────────────────
function generatePuzzle(level = 1) {
  for (let i=0; i<10; i++) {
    const arrows = buildShapeMaze(level);
    if (arrows?.length>0) return buildState(arrows);
  }
  return buildState(buildShapeMaze(level));
}

// Generate shape-based outer border
function generateShapeBorder() {
  const shapes = [
    'rectangle', 'square', 'triangle', 'diamond', 'house', 'arrow', 'cross'
  ];
  const shape = shapes[Math.floor(Math.random() * shapes.length)];
  const border = new Set();
  const margin = 1;
  
  if (shape === 'rectangle' || shape === 'square') {
    // Rectangle/square border
    for (let r = margin; r < ROWS - margin; r++) {
      border.add(ckey(r, margin));
      border.add(ckey(r, COLS - 1 - margin));
    }
    for (let c = margin; c < COLS - margin; c++) {
      border.add(ckey(margin, c));
      border.add(ckey(ROWS - 1 - margin, c));
    }
  } else if (shape === 'triangle') {
    // Triangle border (pointing up)
    const mid = Math.floor(COLS / 2);
    const base = ROWS - 1 - margin;
    for (let r = margin; r <= base; r++) {
      const width = Math.floor((r - margin) * (COLS - 2 * margin) / (base - margin));
      const left = mid - Math.floor(width / 2);
      const right = mid + Math.floor(width / 2);
      border.add(ckey(r, left));
      border.add(ckey(r, right));
    }
    for (let c = margin; c < COLS - margin; c++) {
      border.add(ckey(base, c));
    }
  } else if (shape === 'diamond') {
    // Diamond border
    const midR = Math.floor(ROWS / 2);
    const midC = Math.floor(COLS / 2);
    for (let r = margin; r < ROWS - margin; r++) {
      const dist = Math.abs(r - midR);
      const width = Math.floor((ROWS - 2 * margin) / 2) - dist;
      if (width >= 0) {
        border.add(ckey(r, midC - width));
        border.add(ckey(r, midC + width));
      }
    }
  } else if (shape === 'house') {
    // House shape (triangle on rectangle)
    const roofHeight = Math.floor((ROWS - margin) / 3);
    const mid = Math.floor(COLS / 2);
    // Roof (triangle)
    for (let r = margin; r < margin + roofHeight; r++) {
      const width = Math.floor((r - margin) * (COLS - 2 * margin) / roofHeight);
      const left = mid - Math.floor(width / 2);
      const right = mid + Math.floor(width / 2);
      border.add(ckey(r, left));
      border.add(ckey(r, right));
    }
    // Walls (rectangle)
    const base = ROWS - 1 - margin;
    for (let r = margin + roofHeight; r <= base; r++) {
      border.add(ckey(r, margin));
      border.add(ckey(r, COLS - 1 - margin));
    }
    for (let c = margin; c < COLS - margin; c++) {
      border.add(ckey(base, c));
    }
  } else if (shape === 'arrow') {
    // Arrow pointing right
    const mid = Math.floor(ROWS / 2);
    // Shaft
    for (let c = margin; c < COLS - margin - 2; c++) {
      border.add(ckey(mid, c));
    }
    // Head (triangle)
    const headStart = COLS - margin - 3;
    for (let r = margin; r < ROWS - margin; r++) {
      if (r !== mid) {
        const dist = Math.abs(r - mid);
        if (dist <= 2) border.add(ckey(r, headStart + dist));
      }
    }
    // Tail
    for (let r = margin; r < ROWS - margin; r++) {
      if (r !== mid) border.add(ckey(r, margin));
    }
  } else if (shape === 'cross') {
    // Cross shape
    const midR = Math.floor(ROWS / 2);
    const midC = Math.floor(COLS / 2);
    const armLen = Math.min(ROWS, COLS) / 3;
    // Vertical bar
    for (let r = margin; r < ROWS - margin; r++) {
      if (Math.abs(r - midR) < armLen) {
        border.add(ckey(r, midC));
      }
    }
    // Horizontal bar
    for (let c = margin; c < COLS - margin; c++) {
      if (Math.abs(c - midC) < armLen) {
        border.add(ckey(midR, c));
      }
    }
  }
  
  return border;
}

function buildShapeMaze(level) {
  const filled = new Set();
  const arrows = [];
  const reservedExitCells = new Set();
  const arrowHeads = new Set(); // Track all arrow head cells
  const arrowTails = new Set(); // Track all arrow tail cells
  
  // Level-based parameters
  const minArrowLen = 4 + Math.floor(level / 2);
  const maxArrowLen = 8 + Math.floor(level / 2);
  
  // First, create shape-based border
  const shapeBorder = generateShapeBorder();
  shapeBorder.forEach(k => filled.add(k));
  
  function exitValid(hr, hc, d) {
    let [r, c] = mv(hr, hc, d);
    while (inB(r, c)) {
      const key = ckey(r, c);
      if (!filled.has(key)) return false;
      [r, c] = mv(r, c, d);
    }
    return true;
  }
  
  function canPlaceArrow(hr, hc, tailCell) {
    // Check: head should not touch any tail, tail should not touch any head
    const headKey = ckey(hr, hc);
    const tailKey = ckey(...tailCell);
    
    // Head cannot be at any tail position
    if (arrowTails.has(headKey)) return false;
    
    // Tail cannot be at any head position
    if (arrowHeads.has(tailKey)) return false;
    
    // Head cannot be adjacent to any tail
    for (const d of DIRS) {
      const [nr, nc] = mv(hr, hc, d);
      if (arrowTails.has(ckey(nr, nc))) return false;
    }
    
    // Tail cannot be adjacent to any head
    for (const d of DIRS) {
      const [nr, nc] = mv(...tailCell, d);
      if (arrowHeads.has(ckey(nr, nc))) return false;
    }
    
    return true;
  }
  
  function isFrontier(r, c) {
    const key = ckey(r, c);
    if (filled.has(key) || reservedExitCells.has(key)) return false;
    return DIRS.some(d => {
      const [nr, nc] = mv(r, c, d);
      return !inB(nr, nc) || filled.has(ckey(nr, nc));
    });
  }
  
  let safety = ROWS * COLS * 6;
  while (filled.size < ROWS * COLS && safety-- > 0) {
    const frontier = [];
    for (let r = 0; r < ROWS; r++) {
      for (let c = 0; c < COLS; c++) {
        if (isFrontier(r, c)) frontier.push([r, c]);
      }
    }
    if (!frontier.length) break;
    frontier.sort(() => Math.random() - 0.5);
    
    let placed = false;
    for (const [hr, hc] of frontier) {
      const validExits = DIRS.filter(d => exitValid(hr, hc, d));
      if (!validExits.length) continue;
      
      const exitDir = validExits[Math.floor(Math.random() * validExits.length)];
      const exitRay = new Set();
      let [er, ec] = mv(hr, hc, exitDir);
      while (inB(er, ec)) {
        exitRay.add(ckey(er, ec));
        [er, ec] = mv(er, ec, exitDir);
      }
      
      // Build path
      const cells = [[hr, hc]];
      const bodySet = new Set([ckey(hr, hc)]);
      let curDir = OPP[exitDir];
      const targetLen = minArrowLen + Math.floor(Math.random() * (maxArrowLen - minArrowLen + 1));
      let turns = 0;
      const maxTurns = 3 + Math.floor(level / 2);
      
      for (let s = 0; s < targetLen - 1; s++) {
        const [lr, lc] = cells[cells.length - 1];
        const cands = DIRS.filter(d => {
          if (d === OPP[curDir] && cells.length > 1) return false;
          const [nr, nc] = mv(lr, lc, d);
          const k = ckey(nr, nc);
          return inB(nr, nc) && !filled.has(k) && !bodySet.has(k) && !exitRay.has(k) && !reservedExitCells.has(k);
        });
        
        if (!cands.length) break;
        
        let ch;
        if (turns < maxTurns && cands.length > 1 && Math.random() < 0.5) {
          const turnCands = cands.filter(d => d !== curDir);
          ch = turnCands[Math.floor(Math.random() * turnCands.length)];
          turns++;
        } else {
          ch = (cands.includes(curDir) && Math.random() < 0.6) ? curDir : cands[Math.floor(Math.random() * cands.length)];
        }
        
        const [nr, nc] = mv(lr, lc, ch);
        cells.push([nr, nc]);
        bodySet.add(ckey(nr, nc));
        curDir = ch;
      }
      
      if (cells.length < 2) continue;
      
      // Check head/tail separation
      const tailCell = cells[cells.length - 1];
      if (!canPlaceArrow(hr, hc, tailCell)) continue;
      
      cells.reverse();
      cells.forEach(([r, c]) => filled.add(ckey(r, c)));
      exitRay.forEach(k => reservedExitCells.add(k));
      
      // Track head and tail
      arrowHeads.add(ckey(hr, hc));
      arrowTails.add(ckey(...tailCell));
      
      arrows.push({ id: arrows.length, cells, exitDir, headCell: [hr, hc], cleared: false });
      placed = true;
      break;
    }
    
    if (!placed) {
      // Try single-cell arrows as fallback
      for (const [hr, hc] of frontier) {
        const v = DIRS.filter(d => exitValid(hr, hc, d));
        if (v.length && canPlaceArrow(hr, hc, [hr, hc])) {
          filled.add(ckey(hr, hc));
          arrowHeads.add(ckey(hr, hc));
          arrowTails.add(ckey(hr, hc));
          arrows.push({ id: arrows.length, cells: [[hr, hc]], exitDir: v[Math.floor(Math.random() * v.length)], headCell: [hr, hc], cleared: false });
          break;
        }
      }
    }
  }
  
  return arrows;
}


function buildState(arrows) {
  const cellMap={};
  arrows.forEach(a=>{if(!a.cleared) a.cells.forEach(([r,c])=>{cellMap[ckey(r,c)]=a.id;});});
  return {arrows,cellMap};
}

function canClear(a,arrows,cellMap) {
  if(a.cleared) return false;
  let [r,c]=mv(...a.headCell,a.exitDir);
  while(inB(r,c)){
    const k=ckey(r,c);
    if(k in cellMap&&!arrows[cellMap[k]].cleared) return false;
    [r,c]=mv(r,c,a.exitDir);
  }
  return true;
}

// ─── App ──────────────────────────────────────────────────────────────────────
export default function App() {
  const [level,    setLevel]    = useState(1);
  const [puzzle,   setPuzzle]   = useState(()=>generatePuzzle(1));
  const [lives,    setLives]    = useState(3);
  const [moves,    setMoves]    = useState(0);
  const [score,    setScore]    = useState(0);
  const [streak,   setStreak]   = useState(0);
  const [flash,    setFlash]    = useState(null);
  const [anim,     setAnim]     = useState(null); // {id,dir,dist,start,progress}
  const [solved,   setSolved]   = useState(false);
  const [gameOver, setGameOver] = useState(false);
  const [showHint, setShowHint] = useState(false);
  const timer = useRef(null);
  const rafId = useRef(null);

  const {arrows,cellMap}=puzzle;
  const total   = arrows.length;
  const cleared = arrows.filter(a=>a.cleared).length;
  const readyNow= arrows.filter(a=>!a.cleared&&canClear(a,arrows,cellMap)).length;

  useEffect(()=>{ 
    if(cleared>0&&cleared===total) {
      setSolved(true);
      // Auto-advance to next level after a delay
      setTimeout(()=>{
        setLevel(l=>l+1);
        setPuzzle(generatePuzzle(level+1));
        setLives(3);
        setMoves(0);
        setStreak(0);
        setSolved(false);
        setFlash(null);
        setShowHint(false);
      }, 2000);
    }
  },[cleared,total,level]);

  const handleClick=useCallback((a)=>{
    if(solved||gameOver||a.cleared||anim) return;
    clearTimeout(timer.current);
    const {arrows,cellMap} = puzzle;
    if(canClear(a,arrows,cellMap)){
      const duration = 900;
      const start = performance.now();
      setAnim({id:a.id,progress:0,start});

      const stepFn = (ts)=>{
        setAnim(prev=>{
          if(!prev || prev.id!==a.id) return prev;
          const t = Math.min(1,(ts-start)/duration);
          if(t>=1){
            // Finish: mark arrow cleared and update score
            setPuzzle(prevState=>buildState(prevState.arrows.map(x=>x.id===a.id?{...x,cleared:true}:{...x})));
        setMoves(m=>m+1);
        setStreak(s=>{const ns=s+1;setScore(sc=>sc+10+(ns>2?ns*5:0));return ns;});
            setAnim(null);
            return null;
          }
          return {...prev,progress:t};
        });
        if(rafId.current) rafId.current = requestAnimationFrame(stepFn);
      };
      if(rafId.current) cancelAnimationFrame(rafId.current);
      rafId.current = requestAnimationFrame(stepFn);
    } else {
      setFlash({id:a.id,ok:false});
      setStreak(0);
      timer.current=setTimeout(()=>setFlash(null),360);
      setLives(l=>{const nl=l-1;if(nl<=0)setGameOver(true);return nl;});
    }
  },[puzzle,anim,solved,gameOver,setPuzzle,setLives,setMoves,setScore,setStreak]);

  const restart=()=>{
    clearTimeout(timer.current);
    setLevel(1);
    setPuzzle(generatePuzzle(1));
    setLives(3);setMoves(0);setScore(0);setStreak(0);
    setSolved(false);setGameOver(false);setFlash(null);setShowHint(false);
  };

  const today=new Date().toLocaleDateString("en-US",{month:"long",day:"numeric"});
  const W=PAD*2+COLS*CELL, H=PAD*2+ROWS*CELL;

  return (
    <div style={{
      minHeight:"100vh", background:"#eaeae3",
      display:"flex", flexDirection:"column", alignItems:"center",
      fontFamily:"'Georgia',serif", userSelect:"none", paddingBottom:28,
    }}>
      {/* Header */}
      <div style={{
        width:"100%",maxWidth:500,background:"white",
        padding:"12px 18px 10px",
        display:"flex",alignItems:"center",justifyContent:"space-between",
        borderBottom:"1.5px solid #ddddd0",
        boxShadow:"0 1px 8px rgba(0,0,0,0.07)",
      }}>
        <Pill>🚀 {cleared}</Pill>
        <div style={{fontSize:17,fontWeight:800,color:"#0d1a4a"}}>Level {level}</div>
        <Pill>Normal</Pill>
      </div>

      {/* Lives */}
      <div style={{
        width:"100%",maxWidth:500,background:"white",
        display:"flex",justifyContent:"center",gap:16,padding:"9px 0 7px",
        borderBottom:"1px solid #e8e8de",
      }}>
        {[0,1,2].map(i=>(
          <span key={i} style={{
            fontSize:28,lineHeight:1,
            color:i<lives?"#e53935":"#ddd",
            filter:i<lives?"drop-shadow(0 1px 5px rgba(229,57,53,0.4))":"none",
            transition:"all 0.3s",
          }}>♥</span>
        ))}
      </div>

      {/* Progress */}
      <div style={{
        width:"100%",maxWidth:500,background:"white",
        padding:"7px 18px",borderBottom:"1px solid #e8e8de",
        display:"flex",alignItems:"center",gap:10,
      }}>
        <div style={{flex:1,height:5,background:"#e4e4d8",borderRadius:99,overflow:"hidden"}}>
          <div style={{
            height:"100%",borderRadius:99,
            width:`${(cleared/total)*100}%`,
            background:"linear-gradient(90deg,#0d1a4a,#2a52cc)",
            transition:"width 0.4s ease",
          }}/>
        </div>
        <span style={{fontSize:11,color:"#aaa",minWidth:52,textAlign:"right",fontFamily:"monospace"}}>
          {cleared}/{total}
        </span>
      </div>

      {/* Ready count */}
      <div style={{
        width:"100%",maxWidth:500,background:"#f8f8f2",
        padding:"5px 18px",borderBottom:"1px solid #e8e8de",
        fontSize:12,fontWeight:600,
        color:readyNow>0?"#16a34a":"#dc2626",
        display:"flex",alignItems:"center",gap:5,
      }}>
        <span>{readyNow>0?"✓":"⚠"}</span>
        {readyNow} arrow{readyNow!==1?"s":""} ready to clear
      </div>

      {/* Board */}
      <div style={{
        marginTop:12,background:"white",borderRadius:14,
        border:"1.5px solid #ccc",overflow:"hidden",
        boxShadow:"0 6px 28px rgba(0,0,0,0.12)",lineHeight:0,
      }}>
        <svg width={W} height={H} style={{display:"block"}}>
          {/* Grid */}
          {Array.from({length:ROWS+1},(_,i)=>(
            <line key={`h${i}`}
              x1={PAD} y1={PAD+i*CELL} x2={PAD+COLS*CELL} y2={PAD+i*CELL}
              stroke="#ececE4" strokeWidth="0.7"/>
          ))}
          {Array.from({length:COLS+1},(_,i)=>(
            <line key={`v${i}`}
              x1={PAD+i*CELL} y1={PAD} x2={PAD+i*CELL} y2={PAD+ROWS*CELL}
              stroke="#ececE4" strokeWidth="0.7"/>
          ))}

          {/* Cleared shading */}
          {arrows.filter(a=>a.cleared).map(a=>
            a.cells.map(([r,c])=>(
              <rect key={`s${a.id}-${r}-${c}`}
                x={PAD+c*CELL+1} y={PAD+r*CELL+1}
                width={CELL-2} height={CELL-2}
                fill="#dce8ff" rx={2}/>
            ))
          )}

          {/* Arrows */}
          {arrows.filter(a=>!a.cleared).map(a=>{
            const isFlash=flash?.id===a.id;
            const clearable=canClear(a,arrows,cellMap);
            let color=NAVY;
            if(anim?.id===a.id)             color="#16a34a";
            else if(isFlash)                color=flash.ok?"#16a34a":"#dc2626";
            else if(showHint&&clearable)    color="#1d4ed8";
            return (
              <g key={a.id}>
                <ArrowShape arrow={a} color={color} onClick={()=>handleClick(a)} anim={anim}/>
              </g>
            );
          })}
        </svg>
      </div>

      {/* Stats */}
      <div style={{marginTop:10,display:"flex",gap:22,fontSize:13,fontWeight:700,color:"#333",alignItems:"center"}}>
        <span>{moves} moves</span>
        <span>{score} pts</span>
        {streak>2&&<span style={{color:"#d97706"}}>🔥 ×{streak}</span>}
      </div>

      {/* Buttons */}
      <div style={{marginTop:10,display:"flex",gap:10}}>
        <button onClick={()=>setShowHint(h=>!h)} style={{
          background:showHint?"#1d4ed8":"white",
          color:showHint?"white":"#444",
          border:"1.5px solid #ccc",borderRadius:22,
          padding:"7px 18px",fontSize:13,fontWeight:600,
          cursor:"pointer",transition:"all 0.2s",
        }}>💡 {showHint?"Hints ON":"Hints"}</button>
        <button onClick={restart} style={{
          background:"#0d1a4a",color:"white",
          border:"none",borderRadius:22,padding:"8px 22px",
          fontSize:13,fontWeight:700,cursor:"pointer",
          boxShadow:"0 3px 10px rgba(13,26,74,0.3)",
        }}>New Puzzle</button>
      </div>

      <p style={{marginTop:8,fontSize:11,color:"#bbb",textAlign:"center",maxWidth:270,lineHeight:1.6}}>
        Tap an arrow whose tip can exit the grid without being blocked.
      </p>

      {/* Modal */}
      {(solved||gameOver)&&(
        <div style={{
          position:"fixed",inset:0,background:"rgba(0,0,0,0.6)",
          backdropFilter:"blur(10px)",
          display:"flex",alignItems:"center",justifyContent:"center",zIndex:999,
        }}>
          <div style={{
            background:"white",borderRadius:28,
            padding:"44px 52px",textAlign:"center",
            maxWidth:300,width:"88%",
            boxShadow:"0 24px 80px rgba(0,0,0,0.35)",
          }}>
            <div style={{fontSize:64,marginBottom:12}}>{solved?"🎊":"💔"}</div>
            <div style={{fontSize:26,fontWeight:800,color:"#0d1a4a",marginBottom:6}}>
              {solved?"Puzzle Cleared!":"Game Over"}
            </div>
            <div style={{fontSize:14,color:"#888",marginBottom:22}}>
              {solved?`${moves} moves · ${score} pts`:`Cleared ${cleared} of ${total}`}
            </div>
            <button onClick={restart} style={{
              background:"#0d1a4a",color:"white",border:"none",
              borderRadius:18,padding:"13px 40px",
              fontSize:16,fontWeight:800,cursor:"pointer",
            }}>Play Again</button>
          </div>
        </div>
      )}
    </div>
  );
}

function Pill({children}){
  return(
    <div style={{
      background:"#f0f0e6",borderRadius:22,padding:"5px 14px",
      fontSize:13,fontWeight:700,color:"#444",
      display:"flex",alignItems:"center",gap:4,
    }}>{children}</div>
  );
}
